VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "clsCCSQL"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = False
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = False
Option Compare Database
Option Explicit

'---------------------------------------------------------------------------------------
' Module      : clsCCSQL
' Version     : 1.0
' Author      : Christian Coppes
' Date        : 24.08.2012
' Last Change : 28.10.2012
' Purpose     : Create SQL String from simple properties to emulate SQL in VBA
'---------------------------------------------------------------------------------------

' ######################################################################################
' --------------------------------------------------------------------------------------
' -------------------------------------- Constants -------------------------------------
' --------------------------------------------------------------------------------------
' ######################################################################################

Private Const cMODULENAME As String = "clsCCSQL"

Public Enum EnmOrderBy
    enmOrderASC
    enmOrderDESC
End Enum

Public Enum EnmCondition
    EQUAL_TO
    NOT_EQUAL_TO
    LOWER_THAN
    GREATER_THAN
    LOWER_OR_EQUAL
    GREATER_OR_EQUAL
    IS_NULL
    IS_NOT_NULL
    LIKE_
End Enum

Private Type typFieldValue
    FieldName As String
    FieldValue As String
    TestFieldExists As Boolean
    UseSquareBrackets As Boolean
End Type

' ######################################################################################
' --------------------------------------------------------------------------------------
' ----------------------------------- Class Variables ----------------------------------
' --------------------------------------------------------------------------------------
' ######################################################################################

Private prv_strFields() As String
Private prv_strGroupByFields() As String
Private prv_strInsertFields() As String
Private prv_strOrderByFields() As String
Private prv_intOrderBy() As EnmOrderBy
Private prv_strValues() As String
Private prv_objFieldsValues() As typFieldValue
Private prv_colErrors As Collection
Private prv_colValues As Collection
Private prv_strCommand As String
Private prv_strCommandTable As String
Private prv_strGroupBy As String
Private prv_strOrderBy As String
Private prv_colConditions As Collection
Private prv_SQLString As String
Private prv_bolHaving As Boolean                ' True = HAVING, False = WHERE

' ######################################################################################
' --------------------------------------------------------------------------------------
' ------------------------------------- Properties -------------------------------------
' --------------------------------------------------------------------------------------
' ######################################################################################

Public UseSquareBrackets As Boolean
Public TestFieldExists As Boolean

'---------------------------------------------------------------------------------------
' Property    : CommandTable
' Access Meth.: Read Only
' Date        : 04.11.2012
' Purpose     : Get the name of the selected table or query, with or without square brackets
' Parameters  : bolWithSquareBrackets: True: The table/query name is enclosed in square brackets
' Returns     : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get CommandTable(Optional bolWithSquareBrackets As Boolean = True) As String
    If Not prv_strCommandTable = "" Then
        CommandTable = IIf(bolWithSquareBrackets, "[", "") & prv_strCommandTable & IIf(bolWithSquareBrackets, "]", "")
    Else
        CommandTable = ""
    End If
End Property

'---------------------------------------------------------------------------------------
' Property    : GroupByField
' Access Meth.: Read Only
' Date        : 04.11.2012
' Purpose     : Get the name of a field of the GROUP BY field list
' Parameters  : lngIndex: Number of the field (0 to count of GROUP BY fields -1)
'               bolWithSquareBrackets: True: The field name is enclosed in square brackets
' Returns     : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get GroupByField(lngIndex As Long, Optional bolWithSquareBrackets As Boolean = True) As String
    GroupByField = ""
    If prv_strGroupBy <> "" Then
        If lngIndex <= UBound(prv_strGroupByFields) Then
            If prv_strGroupByFields(lngIndex) <> "" Then
                GroupByField = IIf(bolWithSquareBrackets, "[", "") & prv_strGroupByFields(lngIndex) & IIf(bolWithSquareBrackets, "]", "")
            End If
        End If
    End If
End Property

'---------------------------------------------------------------------------------------
' Property    : GroupByFields
' Access Meth.: Read Only
' Date        : 04.11.2012
' Purpose     : Get the complete array of GROUP BY field names without square brackets
' Returns     : Stringarray
'---------------------------------------------------------------------------------------
'
Public Property Get GroupByFields() As String()
    GroupByFields = prv_strGroupByFields
End Property

'---------------------------------------------------------------------------------------
' Property    : OrderByField
' Access Meth.: Read Only
' Date        : 04.11.2012
' Purpose     : Get the name of a field of the ORDER BY field list
' Parameters  : lngIndex: Number of the field (0 to count of ORDER BY fields -1)
'               bolWithSquareBrackets: True: The field name is enclosed in square brackets
' Returns     : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get OrderByField(lngIndex As Long, Optional bolWithSquareBrackets As Boolean = True) As String
    OrderByField = ""
    If prv_strOrderBy <> "" Then
        If lngIndex <= UBound(prv_strOrderByFields) Then
            If prv_strOrderByFields(lngIndex) <> "" Then
                OrderByField = IIf(bolWithSquareBrackets, "[", "") & prv_strOrderByFields(lngIndex) & _
                               IIf(bolWithSquareBrackets, "]", "") & _
                               IIf(prv_intOrderBy(lngIndex) = enmOrderDESC, " DESC", "")
            End If
        End If
    End If
End Property

'---------------------------------------------------------------------------------------
' Property    : OrderByFields
' Access Meth.: Read Only
' Date        : 04.11.2012
' Purpose     : Get the complete array of ORDER BY field names without square brackets
' Returns     : Stringarray
'---------------------------------------------------------------------------------------
'
Public Property Get OrderByFields() As String()
    OrderByFields = prv_strOrderByFields
End Property

'---------------------------------------------------------------------------------------
' Property    : Field
' Access Meth.: Read Only
' Date        : 04.11.2012
' Purpose     : Get the name of a SELECT field
' Parameters  : lngIndex: Number of the field (0 to count of SELECT fields -1)
'               bolWithSquareBrackets: True: The field name is enclosed in square brackets
'                                      If the field name is "*", "NULL" or begins with "(" this option is ignored.
' Returns     : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get Field(lngIndex As Long, Optional bolWithSquareBrackets As Boolean = True) As String
    Field = ""
    If lngIndex <= UBound(prv_strFields) Then
        If prv_strFields(lngIndex) <> "" Then
            If prv_strFields(lngIndex) <> "NULL" And prv_strFields(lngIndex) <> "*" And Left(prv_strFields(lngIndex), 1) <> "(" Then
                Field = IIf(bolWithSquareBrackets, "[", "") & prv_strFields(lngIndex) & _
                        IIf(bolWithSquareBrackets, "]", "")
            Else
                Field = prv_strFields(lngIndex)
            End If
        End If
    End If
End Property

'---------------------------------------------------------------------------------------
' Property    : Fields
' Access Meth.: Read Only
' Date        : 04.11.2012
' Purpose     : Get the complete array of SELECT field names without square brackets
' Returns     : Stringarray
'---------------------------------------------------------------------------------------
'
Public Property Get Fields() As String()
    Fields = prv_strFields
End Property

'---------------------------------------------------------------------------------------
' Property    : Condition
' Access Meth.: Read Only
' Date        : 04.11.2012
' Purpose     : Get a condition object (WHERE, HAVING, AND, OR...)
'               As it returns the object reference, changing values in the returned object is possible for read AND write
'               Can be used to manipulate the condition objects before using "SQLString"
' Parameters  : lngIndex: 1 to count of condition objects, otherwise returns an empty conditon object
'               bolWithSquareBrackets: True: The field name is enclosed in square brackets, CHANGES THE OBJECT !
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Property Get Condition(lngIndex As Long, Optional bolWithSquareBrackets As Boolean = True) As clsCCSQL_Condition
    Dim objCondition As clsCCSQL_Condition
    If Not prv_colConditions Is Nothing Then
        If lngIndex >= 1 And lngIndex <= prv_colConditions.Count Then
            Set objCondition = prv_colConditions(lngIndex)
            objCondition.UseSquareBrackets = bolWithSquareBrackets
        End If
    End If
    Set Condition = objCondition
End Property

' ######################################################################################
' --------------------------------------------------------------------------------------
' -------------------------------------- Methods ---------------------------------------
' --------------------------------------------------------------------------------------
' ######################################################################################

'---------------------------------------------------------------------------------------
' Procedure   : SQLString
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Output a dynamic SQL string using the selected commands
' Parameters  : - (all values must be set with the "commands" like "SELECT_", "WHERE_", "FROM_" and so on, see below)
' Returns     : str - String
'---------------------------------------------------------------------------------------
'
Public Function SQLString() As String
    Dim varCondition As Variant
    Dim i As Long
    Dim db As DAO.Database
    Dim td As DAO.TableDef
    Dim qd As DAO.QueryDef
    Dim fld As DAO.Field
    Dim strFieldList As String
    Dim strValueList As String
    Dim objFieldValue As typFieldValue
    Dim objCondition As clsCCSQL_Condition
    Dim bolAddGroupBy As Boolean

    If prv_strCommand = "" Then AddError "No command was chosen. Use SELECT_, INSERT_, DELETE_ or UPDATE_ first."
    If Me.CommandTable(False) = "" Then AddError "No table/query was specified. Use FROM_, INSERT_, DELETE_ or UPDATE_ first."

    If prv_colErrors.Count = 0 Then
        Select Case prv_strCommand
            Case "SELECT"
                If Join(prv_strFields, "") = "" Then AddError "No field list was specified."
                If prv_colErrors.Count = 0 Then
                    For i = 0 To UBound(prv_strFields)
                        strFieldList = strFieldList & Me.Field(i, Me.UseSquareBrackets) & ","
                    Next
                    If Right(strFieldList, 1) = "," Then strFieldList = Left(strFieldList, Len(strFieldList) - 1)
                    prv_SQLString = prv_strCommand & " " & strFieldList & " FROM " & Me.CommandTable(Me.UseSquareBrackets) & " "

                    bolAddGroupBy = False
                    For Each objCondition In prv_colConditions
                        If objCondition.HavingCondition = True And bolAddGroupBy = False Then
                            bolAddGroupBy = True
                            prv_SQLString = prv_SQLString & prv_strGroupBy
                        End If
                        prv_SQLString = prv_SQLString & objCondition.ConditionString
                    Next
                    If bolAddGroupBy = False And Left(prv_strGroupBy, 9) = " GROUP BY" Then
                        prv_SQLString = prv_SQLString & prv_strGroupBy
                    End If
                    If prv_strOrderBy <> "" Then
                        strFieldList = ""
                        prv_SQLString = prv_SQLString & prv_strOrderBy
                        For i = 0 To UBound(prv_strOrderByFields)
                            strFieldList = strFieldList & Me.OrderByField(i, Me.UseSquareBrackets) & ","
                        Next
                        If Right(strFieldList, 1) = "," Then strFieldList = Left(strFieldList, Len(strFieldList) - 1)
                        prv_SQLString = prv_SQLString & strFieldList
                    End If
                End If
            Case "UPDATE", "UPDATEQRY"
                If prv_objFieldsValues(0).FieldName = "" Then AddError "No field/value pairs where specified. Use SET_ first."
                If prv_colErrors.Count = 0 Then
                    Set db = CurrentDb
                    If Len(prv_strCommand) = 9 Then
                        Set qd = db.QueryDefs(Me.CommandTable(Me.UseSquareBrackets))
                    Else
                        Set td = db.TableDefs(Me.CommandTable(Me.UseSquareBrackets))
                    End If
                    For i = 0 To UBound(prv_objFieldsValues)
                        objFieldValue = prv_objFieldsValues(i)
                        With objFieldValue
                            If .TestFieldExists Then
                                On Error Resume Next
                                If Len(prv_strCommand) = 9 Then
                                    Set fld = qd.Fields(.FieldName)
                                Else
                                    Set fld = td.Fields(.FieldName)
                                End If
                                On Error GoTo 0
                                If Err.Number <> 0 Then
                                    AddError "Field " & .FieldName & " does not exist in the table/query " & Me.CommandTable(False) & " !"
                                End If
                            End If
                        End With
                    Next
                    If prv_colErrors.Count = 0 Then
                        prv_SQLString = "UPDATE " & Me.CommandTable(Me.UseSquareBrackets)
                        i = 0
                        For i = 0 To UBound(prv_objFieldsValues)
                            objFieldValue = prv_objFieldsValues(i)
                            With objFieldValue
                                If i = 0 Then prv_SQLString = prv_SQLString & "  SET "
    
                                If .TestFieldExists Then
                                    If Len(prv_strCommand) = 9 Then
                                        Set fld = qd.Fields(.FieldName)
                                    Else
                                        Set fld = td.Fields(.FieldName)
                                    End If
                                    If .UseSquareBrackets Then
                                        If Left(.FieldName, 1) <> "[" Then .FieldName = "[" & .FieldName
                                        If Right(.FieldName, 1) <> "]" Then .FieldName = .FieldName & "]"
                                    End If
                                    Select Case fld.Type
                                        Case dbDate
                                            If IsDate(.FieldValue) Then
                                                prv_SQLString = prv_SQLString & .FieldName & " = " & Format(CDate(.FieldValue), "\#m\/d\/yyyy\#") & ", "
                                            Else
                                                If .FieldValue = "NULL" Then
                                                    prv_SQLString = prv_SQLString & .FieldName & " = NULL, "
                                                Else
                                                    AddError "Wrong data for date field: " & .FieldValue
                                                    Exit For
                                                End If
                                            End If
                                        Case dbChar, dbText, dbMemo
                                            prv_SQLString = prv_SQLString & .FieldName & " = '" & .FieldValue & "', "
                                        Case Else
                                            prv_SQLString = prv_SQLString & .FieldName & " = " & .FieldValue & ", "
                                    End Select
                                Else
                                    If .UseSquareBrackets Then
                                        If Left(.FieldName, 1) <> "[" Then .FieldName = "[" & .FieldName
                                        If Right(.FieldName, 1) <> "]" Then .FieldName = .FieldName & "]"
                                    End If
                                    prv_SQLString = prv_SQLString & .FieldName & " = " & .FieldValue & ", "
                                End If
                                i = i + 1
                            End With
                        Next
                        If Right(prv_SQLString, 2) = ", " Then prv_SQLString = Left(prv_SQLString, Len(prv_SQLString) - 2) & " "
                        
                        For Each objCondition In prv_colConditions
                            If Not objCondition.HavingCondition Then
                                prv_SQLString = prv_SQLString & objCondition.ConditionString
                            End If
                        Next
                    End If
                End If
            Case "INSERT", "INSERTQRY"
                If prv_objFieldsValues(0).FieldName = "" Then AddError "No field/value pairs where specified. Use SET_ first."
                If prv_colErrors.Count = 0 Then
                    Set db = CurrentDb
                    If Len(prv_strCommand) = 9 Then
                        Set qd = db.QueryDefs(Me.CommandTable(Me.UseSquareBrackets))
                    Else
                        Set td = db.TableDefs(Me.CommandTable(Me.UseSquareBrackets))
                    End If
                    For i = 0 To UBound(prv_objFieldsValues)
                        objFieldValue = prv_objFieldsValues(i)
                        With objFieldValue
                            If .TestFieldExists Then
                                On Error Resume Next
                                If Len(prv_strCommand) = 9 Then
                                    Set fld = qd.Fields(.FieldName)
                                Else
                                    Set fld = td.Fields(.FieldName)
                                End If
                                On Error GoTo 0
                                If Err.Number <> 0 Then
                                    AddError "Field " & .FieldName & " does not exist in the table/query " & Me.CommandTable(False) & " !"
                                End If
                            End If
                        End With
                    Next
                    If prv_colErrors.Count = 0 Then
                        i = 0
                        For i = 0 To UBound(prv_objFieldsValues)
                            objFieldValue = prv_objFieldsValues(i)
                            With objFieldValue
                                If .TestFieldExists Then
                                    If Len(prv_strCommand) = 9 Then
                                        Set fld = qd.Fields(.FieldName)
                                    Else
                                        Set fld = td.Fields(.FieldName)
                                    End If
                                    If .UseSquareBrackets Then
                                        If Left(.FieldName, 1) <> "[" Then .FieldName = "[" & .FieldName
                                        If Right(.FieldName, 1) <> "]" Then .FieldName = .FieldName & "]"
                                    End If
                                    strFieldList = strFieldList & .FieldName & ","
                                    If .FieldValue = "NULL" Then
                                        strValueList = strValueList & "NULL,"
                                    Else
                                        Select Case fld.Type
                                            Case dbDate
                                                If IsDate(.FieldValue) Then
                                                    strValueList = strValueList & Format(CDate(.FieldValue), "\#m\/d\/yyyy\#") & ","
                                                Else
                                                    AddError "Wrong data for date field: " & .FieldValue
                                                    Exit For
                                                End If
                                            Case dbChar, dbText, dbMemo
                                                strValueList = strValueList & "'" & .FieldValue & "',"
                                            Case Else
                                                strValueList = strValueList & .FieldValue & ","
                                        End Select
                                    End If
                                Else
                                    If .UseSquareBrackets Then
                                        If Left(.FieldName, 1) <> "[" Then .FieldName = "[" & .FieldName
                                        If Right(.FieldName, 1) <> "]" Then .FieldName = .FieldName & "]"
                                    End If
                                    strFieldList = strFieldList & .FieldName & ","
                                    strValueList = strValueList & .FieldValue & ","
                                End If
                                i = i + 1
                            End With
                        Next
                        If Right(strFieldList, 1) = "," Then strFieldList = Left(strFieldList, Len(strFieldList) - 1)
                        If Right(strValueList, 1) = "," Then strValueList = Left(strValueList, Len(strValueList) - 1)
                        If Right(prv_SQLString, 2) = ", " Then prv_SQLString = Left(prv_SQLString, Len(prv_SQLString) - 2) & " "
                        prv_SQLString = "INSERT INTO " & Me.CommandTable(Me.UseSquareBrackets) & " (" & strFieldList & ") VALUES (" & strValueList & ") "
                    End If
                End If
            Case "DELETE"
                Set db = CurrentDb
                On Error Resume Next
                Set td = db.TableDefs(Me.CommandTable(Me.UseSquareBrackets))
                If Err.Number = 0 Then
                    On Error GoTo 0
                    prv_SQLString = "DELETE * FROM " & Me.CommandTable(Me.UseSquareBrackets) & " "
                    
                    For Each objCondition In prv_colConditions
                        If objCondition.HavingCondition = False Then
                            prv_SQLString = prv_SQLString & objCondition.ConditionString
                        End If
                    Next
                Else
                    AddError "Table name does not exist. DELETE can only be used with tables."
                End If
        End Select
    End If
    SQLString = prv_SQLString
End Function

'---------------------------------------------------------------------------------------
' Procedure   : AddError
' Date        : 24.08.2012
' Purpose     : Add an error string to the error collection for later output.
'               Creates the error collection if it does not exist.
' Parameters  : strError: Errorstring of any content
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Private Sub AddError(strError As String)
    If prv_colErrors Is Nothing Then Set prv_colErrors = New Collection
    prv_colErrors.Add strError
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : SELECT_
' Date        : 24.11.2012
' Last Change : 04.11.2012
' Purpose     : "SELECT" command
' Parameters  : varField must be a list of strings containing the wanted fields or expressions
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub SELECT_(ParamArray varField() As Variant)
    Dim varCurrentField As Variant
    Dim i As Long
    
    If UBound(varField) + 1 = 0 Then
        AddError "Please specify which fields you want to select!"
    Else
        If IsArray(varField(0)) Then
            If UBound(varField) = 0 Then varField = varField(0)            ' Copy an array to "varField" so it can used like a normal ParamArray
        End If
        procResetVars
        prv_strCommand = "SELECT"
        ReDim prv_strFields(UBound(varField))
        
        For Each varCurrentField In varField
            If VarType(varCurrentField) <> vbString Then
                AddError "Field list contains a field which is no string, field name: " & CStr(Nz(varCurrentField))
            Else
                ' Save field names without brackets
                If Left(varCurrentField, 1) = "[" Then varCurrentField = Right(varCurrentField, Len(varCurrentField) - 1)
                If Right(varCurrentField, 1) = "]" Then varCurrentField = Left(varCurrentField, Len(varCurrentField) - 1)
                prv_strFields(i) = varCurrentField
            End If
            i = i + 1
        Next
    End If
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : UPDATE_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : "UPDATE" command for tables, set the command to "UPDATE" and reset all other values
' Parameters  : intTable: A table name enum number
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub UPDATE_(intTable As EnmMyTables)
    procResetVars
    prv_strCommand = "UPDATE"
    ' Save table name always without square brackets
    prv_strCommandTable = mID(Tab_Name(intTable), 2, Len(Tab_Name(intTable)) - 2)
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : UPDATE_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : "UPDATE" command for queries, set the command to "UPDATE" and reset all other values
' Parameters  : intQuery: A query name enum number
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub UPDATEQRY_(intQuery As EnmMyQueries)
    procResetVars
    prv_strCommand = "UPDATEQRY"
    ' Save query name always without square brackets
    prv_strCommandTable = mID(Qry_Name(intQuery), 2, Len(Qry_Name(intQuery)) - 2)
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : INSERT_INTO
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : "INSERT" command for tables, set the command to "INSERT" and reset all other values
' Parameters  : intTable: A table name enum number
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub INSERT_INTO(intTable As EnmMyTables)
    procResetVars
    prv_strCommand = "INSERT"
    ' Save table name always without square brackets
    prv_strCommandTable = mID(Tab_Name(intTable), 2, Len(Tab_Name(intTable)) - 2)
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : INSERT_INTO_QRY
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : "INSERT" command for queries, set the command to "INSERT" and reset all other values
' Parameters  : intQuery: A query name enum number
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub INSERT_INTO_QRY(intQuery As EnmMyQueries)
    procResetVars
    prv_strCommand = "INSERTQRY"
    ' Save query name always without square brackets
    prv_strCommandTable = mID(Qry_Name(intQuery), 2, Len(Qry_Name(intQuery)) - 2)
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : DELETE_
' Date        : 24.08.2012
' Purpose     : "DELETE" command for tables only, set the command to "INSERT" and reset all other values
' Parameters  : No parameter, table name must be set with "FROM_"
' Returns     : -
'---------------------------------------------------------------------------------------
'
Public Sub DELETE_()
    procResetVars
    prv_strCommand = "DELETE"
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : FROM_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : "FROM" for "SELECT" and "DELETE", sets the "CommandTable" property to a table name
' Parameters  : intFROM: A table name enum number
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub FROM_(intFROM As EnmMyTables)
    ' Save table name always without square brackets
    prv_strCommandTable = mID(Tab_Name(intFROM), 2, Len(Tab_Name(intFROM)) - 2)
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : FROMQRY_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : "FROMQRY" for "SELECT" and "DELETE", sets the "CommandTable" property to a query name
' Parameters  : intFROM: A query name enum number
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub FROMQRY_(intFROM As EnmMyQueries)
    If prv_strCommand = "DELETE" Then
        AddError """FROMQRY_"" cannot be used with ""DELETE"", use ""FROM_"" instead!"
    Else
        ' Save query name always without square brackets
        prv_strCommandTable = mID(Qry_Name(intFROM), 2, Len(Qry_Name(intFROM)) - 2)
    End If
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : WHERE_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Sets the first condition of a "WHERE" command
' Parameters  : strCmpLeft: String which contains the field name to compare
'               intCondition: Condition like ">","<","="...
'               strCmpRight: Value the field should be compared with
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function WHERE_(strCmpLeft As String, intCondition As EnmCondition, strCmpRight As String) As clsCCSQL_Condition
    If prv_strCommandTable = "" Then
        AddError "Add a table or query name first by using ""FROM_"" or ""FROMQRY_"""
    Else
        Set WHERE_ = fnAddCondition(strCmpLeft, intCondition, strCmpRight, "WHERE")
    End If
End Function

'---------------------------------------------------------------------------------------
' Procedure   : WHERE_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Sets the first condition of a "HAVING" command
' Parameters  : strCmpLeft: String which contains the field name to compare
'               intCondition: Condition like ">","<","="...
'               strCmpRight: Value the field should be compared with
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function HAVING_(strCmpLeft As String, intCondition As EnmCondition, strCmpRight As String) As clsCCSQL_Condition
    If prv_strCommandTable = "" Then
        AddError "Add a table or query name first using ""FROM_"" or ""FROMQRY_"""
    Else
        If prv_strGroupBy = "" Then
            AddError "Use ""GROUPBY_"" first before using ""HAVING_""!"
        Else
            prv_bolHaving = True
            Set HAVING_ = fnAddCondition(strCmpLeft, intCondition, strCmpRight, "HAVING")
        End If
    End If
End Function

'---------------------------------------------------------------------------------------
' Procedure   : AND_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Adds a condition to the last used condition clause ("WHERE" or "HAVING")
' Parameters  : strCmpLeft: String which contains the field name to compare
'               intCondition: Condition like ">","<","="...
'               strCmpRight: Value the field should be compared with
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function AND_(strCmpLeft As String, intCondition As EnmCondition, strCmpRight As String) As clsCCSQL_Condition
    Set AND_ = fnAddCondition(strCmpLeft, intCondition, strCmpRight, "AND")
End Function

'---------------------------------------------------------------------------------------
' Procedure   : AND_begin
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Adds a condition to the last used condition clause ("WHERE" or "HAVING")
'               and opens a bracket which must be closed using "AND_end"
' Parameters  : strCmpLeft: String which contains the field name to compare
'               intCondition: Condition like ">","<","="...
'               strCmpRight: Value the field should be compared with
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function AND_begin(strCmpLeft As String, intCondition As EnmCondition, strCmpRight As String) As clsCCSQL_Condition
    Set AND_begin = fnAddCondition(strCmpLeft, intCondition, strCmpRight, "ANDbegin")
End Function

'---------------------------------------------------------------------------------------
' Procedure   : AND_end
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : closes the last "AND_begin" clause
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function AND_end() As clsCCSQL_Condition
    Set AND_end = fnAddCondition("", 0, "", "ANDend")
End Function

'---------------------------------------------------------------------------------------
' Procedure   : OR_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Adds a condition to the last used condition clause ("WHERE" or "HAVING")
' Parameters  : strCmpLeft: String which contains the field name to compare
'               intCondition: Condition like ">","<","="...
'               strCmpRight: Value the field should be compared with
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function OR_(strCmpLeft As String, intCondition As EnmCondition, strCmpRight As String) As clsCCSQL_Condition
    Set OR_ = fnAddCondition(strCmpLeft, intCondition, strCmpRight, "OR ")
End Function

'---------------------------------------------------------------------------------------
' Procedure   : OR_begin
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Adds a condition to the last used condition clause ("WHERE" or "HAVING")
'               and opens a bracket which must be closed using "OR_end"
' Parameters  : strCmpLeft: String which contains the field name to compare
'               intCondition: Condition like ">","<","="...
'               strCmpRight: Value the field should be compared with
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function OR_begin(strCmpLeft As String, intCondition As EnmCondition, strCmpRight As String) As clsCCSQL_Condition
    Set OR_begin = fnAddCondition(strCmpLeft, intCondition, strCmpRight, "OR begin")
End Function

'---------------------------------------------------------------------------------------
' Procedure   : OR_end
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : closes the last "OR_begin" clause
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function OR_end() As clsCCSQL_Condition
    Set OR_end = fnAddCondition("", 0, "", "OR end")
End Function

'---------------------------------------------------------------------------------------
' Procedure   : NOT_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Adds a condition to the last used condition clause ("WHERE" or "HAVING")
' Parameters  : strCmpLeft: String which contains the field name to compare
'               intCondition: Condition like ">","<","="...
'               strCmpRight: Value the field should be compared with
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function NOT_(strCmpLeft As String, Optional intCondition As EnmCondition, Optional strCmpRight As String) As clsCCSQL_Condition
    Set NOT_ = fnAddCondition(strCmpLeft, intCondition, strCmpRight, "NOT")
End Function

'---------------------------------------------------------------------------------------
' Procedure   : NOT_begin
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Adds a condition to the last used condition clause ("WHERE" or "HAVING")
'               and opens a bracket which must be closed using "NOT_end"
' Parameters  : strCmpLeft: String which contains the field name to compare
'               intCondition: Condition like ">","<","="...
'               strCmpRight: Value the field should be compared with
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function NOT_begin(strCmpLeft As String, intCondition As EnmCondition, strCmpRight As String) As clsCCSQL_Condition
    Set NOT_begin = fnAddCondition(strCmpLeft, intCondition, strCmpRight, "NOTbegin")
End Function

'---------------------------------------------------------------------------------------
' Procedure   : OR_end
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : closes the last "NOT_begin" clause
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Public Function NOT_end() As clsCCSQL_Condition
    Set NOT_end = fnAddCondition("", 0, "", "NOTend")
End Function

'---------------------------------------------------------------------------------------
' Procedure   : GROUPBY_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Adds a "GROUP BY" clause
' Parameters  : varField: A list of fields as strings
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub GROUPBY_(ParamArray varField() As Variant)
    Dim varCurrentField As Variant
    Dim strFields As String
    Dim i As Long
    
    If UBound(varField) + 1 = 0 Then
        AddError "Please specify the fields you want to group!"
    Else
        If IsArray(varField(0)) Then
            If UBound(varField) = 0 Then varField = varField(0)            ' Copy an array to "varField" so it can used like a normal ParamArray
        End If
        ReDim prv_strGroupByFields(UBound(varField))
        
        For Each varCurrentField In varField
            If VarType(varCurrentField) <> vbString Then
                AddError "Field list contains a field which is no string, field name: " & CStr(Nz(varCurrentField))
            Else
                If Nz(varCurrentField) = "" Then
                    AddError "Fieldname " & i & " is empty!"
                Else
                    If Left(varCurrentField, 1) = "[" Then varCurrentField = Right(varCurrentField, Len(varCurrentField) - 1)
                    If Right(varCurrentField, 1) = "]" Then varCurrentField = Left(varCurrentField, Len(varCurrentField) - 1)
                    prv_strGroupByFields(i) = varCurrentField
                End If
            End If
            i = i + 1
        Next
    End If
    prv_strGroupBy = " GROUP BY "
    strFields = Join(prv_strGroupByFields, ",")

    If TestFieldExists Then
        If Not fnFieldsInFieldlist(strFields) Then
            AddError "The GROUP BY field list: " & vbCr & strFields & vbCr & "contains fields not available in the field list:" & vbCr & Join(prv_strFields, ",")
        End If
    End If
    
    If UseSquareBrackets Then
        prv_strGroupBy = prv_strGroupBy & "[" & Join(prv_strGroupByFields, "],[") & "] "
    Else
        prv_strGroupBy = prv_strGroupBy & Join(prv_strGroupByFields, ",") & " "
    End If
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : ORDERBY_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Adds an "ORDER BY" clause
' Parameters  : strField: One field for the ORDER BY clause, add further fields with "THENBY_"
'               intOrderBy: ASC or DESC from EnmOrderBy
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub ORDERBY_(strField As String, Optional intOrderBy As EnmOrderBy = enmOrderASC)
    ReDim prv_strOrderByFields(0)
    ReDim prv_intOrderBy(0)
    ' Save field names always without square brackets
    If Left(strField, 1) = "[" Then strField = Right(strField, Len(strField) - 1)
    If Right(strField, 1) = "]" Then strField = Left(strField, Len(strField) - 1)
    prv_strOrderByFields(0) = strField
    prv_strOrderBy = " ORDER BY "
    prv_intOrderBy(0) = intOrderBy
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : THENBY_
' Date        : 24.08.2012
' Last Change : 04.11.2012
' Purpose     : Adds a field to the "ORDER BY" clause
' Parameters  : strField: One field for the ORDER BY clause, add further fields with another "THENBY_"
'               intOrderBy: ASC or DESC from EnmOrderBy
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub THENBY_(strField As String, Optional intOrderBy As EnmOrderBy = enmOrderASC)
    Dim i As Long
    If prv_strOrderBy = "" Then
        AddError "Please use ""ORDERBY_"" before adding further fields to the ""ORDER BY"" list!"
    Else
        i = UBound(prv_strOrderByFields) + 1
        ReDim Preserve prv_strOrderByFields(i)
        ReDim Preserve prv_intOrderBy(i)
        ' Save field names always without square brackets
        If Left(strField, 1) = "[" Then strField = Right(strField, Len(strField) - 1)
        If Right(strField, 1) = "]" Then strField = Left(strField, Len(strField) - 1)
        prv_strOrderByFields(i) = strField
        prv_intOrderBy(i) = intOrderBy
    End If
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : SET_
' Date        : 04.10.2012
' Purpose     : Used for INSERT and UPDATE to set a field with a value.
'               Adds a SET operation to the SET list.
' Parameters  : strFieldName: Name of the field whose value should be changed
'               strValue: Value which should be set into the field
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub SET_(strFieldName As String, strValue As String)
    Dim i As Long
    Dim objFieldValue As typFieldValue
    
    With objFieldValue
        If Not Me.UseSquareBrackets Then
            If Left(strFieldName, 1) = "[" Then strFieldName = Right(strFieldName, Len(strFieldName) - 1)
            If Right(strFieldName, 1) = "]" Then strFieldName = Left(strFieldName, Len(strFieldName) - 1)
        End If
        .FieldName = strFieldName
        .FieldValue = strValue
        .TestFieldExists = Me.TestFieldExists
        .UseSquareBrackets = Me.UseSquareBrackets
    End With

    If prv_objFieldsValues(0).FieldName = "" Then
        prv_objFieldsValues(0) = objFieldValue
    Else
        i = UBound(prv_objFieldsValues) + 1
        ReDim Preserve prv_objFieldsValues(i)
        prv_objFieldsValues(i) = objFieldValue
    End If
End Sub

'---------------------------------------------------------------------------------------
' Procedure   : fnAddCondition
' Date        : 04.10.2012
' Purpose     : Add a condition object to the condition collection which will be filled
'               from the parameter values and other properties
' Parameters  : strCmpLeft: The left part of the comparison
'               intCondition: The comparison condition like "=" or "IS"
'               strCmpRight: The value for the comparison
'               strOperation: Operations like "WHERE", "AND", "HAVING"
' Returns     : obj - clsCCSQL_Condition
'---------------------------------------------------------------------------------------
'
Private Function fnAddCondition(strCmpLeft As String, intCondition As EnmCondition, strCmpRight As String, _
                                strOperation As String) As clsCCSQL_Condition
    Dim i As Long
    Dim objCond As clsCCSQL_Condition
    Dim objFieldValue As typFieldValue
    Dim varAddCond As Variant
        
    Set objCond = New clsCCSQL_Condition
    With objCond
        If Not Me.UseSquareBrackets Then
            If Left(strCmpLeft, 1) = "[" Then strCmpLeft = Right(strCmpLeft, Len(strCmpLeft) - 1)
            If Right(strCmpLeft, 1) = "]" Then strCmpLeft = Left(strCmpLeft, Len(strCmpLeft) - 1)
        End If
        .FieldName = strCmpLeft
        .FieldValue = strCmpRight
        .UseSquareBrackets = Me.UseSquareBrackets
        .TestFieldExists = Me.TestFieldExists
        Select Case intCondition
            Case EQUAL_TO
                .Condition = "="
            Case NOT_EQUAL_TO
                .Condition = "<>"
            Case LOWER_THAN
                .Condition = "<"
            Case GREATER_THAN
                .Condition = ">"
            Case LOWER_OR_EQUAL
                .Condition = "<="
            Case GREATER_OR_EQUAL
                .Condition = ">="
            Case IS_NULL
                .Condition = "IS"
                .FieldValue = "NULL"
            Case IS_NOT_NULL
                .Condition = "IS NOT"
                .FieldValue = "NULL"
            Case LIKE_
                .Condition = "LIKE"
        End Select
        .Operation = strOperation
        .HavingCondition = prv_bolHaving
    End With
    
    prv_colConditions.Add objCond
    
    Set fnAddCondition = objCond
End Function

'---------------------------------------------------------------------------------------
' Procedure   : fnFieldsInFieldlist
' Date        : 04.10.2012
' Purpose     : Compares a comma separated field list against the field list chosen with SELECT
' Parameters  : strFields: Comma separated field list
' Returns     : bol - Boolean: True if the lists are equal
'---------------------------------------------------------------------------------------
'
Private Function fnFieldsInFieldlist(strFields As String) As Boolean
    Dim i As Long, k As Long
    Dim strFieldTest() As String
    Dim bolFound As Boolean
    Dim bolNotFound As Boolean

    strFieldTest = Split(strFields, ",")
    For i = 0 To UBound(strFieldTest)
        For k = 0 To UBound(prv_strFields)
            If strFieldTest(i) = prv_strFields(k) Then
                bolFound = True
                Exit For
            End If
        Next
        If bolFound = False Then
            bolNotFound = True
            Exit For
        Else
            bolFound = False
        End If
    Next

    fnFieldsInFieldlist = Not bolNotFound
End Function

'---------------------------------------------------------------------------------------
' Procedure   : GetErrorList
' Date        : 04.10.2012
' Purpose     : Returns the error list which all commands of this object produced with "AddError"
' Returns     : str - String
'---------------------------------------------------------------------------------------
'
Public Function GetErrorList() As String
    Dim varError As Variant
    Dim strResult As String
    For Each varError In prv_colErrors
        strResult = strResult & varError & vbCr
    Next
    GetErrorList = strResult
End Function

'---------------------------------------------------------------------------------------
' Procedure   : procResetVars
' Date        : 04.10.2012
' Purpose     : Resets all variables in this object
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Private Sub procResetVars()
    Set prv_colErrors = New Collection
    Set prv_colConditions = New Collection
    ReDim prv_objWhereConditions(0)
    ReDim prv_objHavingConditions(0)
    ReDim prv_strFields(0)
    ReDim prv_strValues(0)
    ReDim prv_objFieldsValues(0)
    ReDim prv_strInsertFields(0)
    ReDim prv_strGroupByFields(0)
    ReDim prv_strOrderByFields(0)
    ReDim prv_intOrderBy(0)
    prv_bolHaving = False
    prv_strCommand = ""
    prv_strCommandTable = ""
    prv_strGroupBy = ""
    prv_strOrderBy = ""
    prv_SQLString = ""
    Me.UseSquareBrackets = True
    Me.TestFieldExists = True
End Sub

' ######################################################################################
' --------------------------------------------------------------------------------------
' ------------------------------ Constructor / Destructor ------------------------------
' --------------------------------------------------------------------------------------
' ######################################################################################

Private Sub Class_Initialize()
    procResetVars
End Sub

Private Sub Class_Terminate()
    Set prv_colErrors = Nothing
End Sub
